<?php

namespace App;

use DB;
use Illuminate\Database\Eloquent\Model;

use function PHPSTORM_META\map;

class BusinessLocation extends Model
{
    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'featured_products' => 'array',
    ];

    /**
     * Return list of locations for a business
     *
     * @param  int  $business_id
     * @param  bool  $show_all = false
     * @param  array  $receipt_printer_type_attribute =
     * @return array
     */
    public static function forDropdown($business_id, $show_all = false, $receipt_printer_type_attribute = false, $append_id = true, $check_permission = true)
    {
        $query = BusinessLocation::where('business_locations.business_id', $business_id)
        ->leftJoin('tax_rates', 'tax_rates.id', '=', 'business_locations.tax')
        ->select('business_locations.id')->Active();
        // dd($query->get());
        /* if ($check_permission) {
            $permitted_locations = auth()->user()->permitted_locations();
            // dd($permitted_locations);
            if ($permitted_locations != 'all') {
                $query->whereIn('id', $permitted_locations);
            }
        } */
        
        $check_model_has_permissions = ModelHasPermissions::with('permissions')->where('model_id', auth()->user()->id)->get()->toArray();        
        // dd($check_model_has_permissions);
        $user = auth()->user();        
        $is_admin = $user->roles[0]->name == 'Admin#'.$user->business_id ? true : false;         
        if($check_model_has_permissions && !$is_admin) {
            $whereIn = [];            
            foreach($query->get() as $v) {                
                foreach($check_model_has_permissions as $cm) {
                    $location = 'location.'.$v->id;                    
                    if($location == $cm['permissions'][0]['name']) {
                        // dd($location_, $cm['permissions'][0]['name']);
                        array_push($whereIn, $v->id);
                    }
                }
            }
            $query->whereIn('business_locations.id', $whereIn);
        }        
        

        if ($append_id) {
            $query->select(
                // DB::raw("IF(location_id IS NULL OR location_id='', name, CONCAT(name, ' (', location_id, ')')) AS name"),
                // \DB::raw("IF(location_id IS NULL OR location_id='', name, name)"),
                'business_locations.name',
                'business_locations.id',
                'business_locations.receipt_printer_type',
                'business_locations.selling_price_group_id',
                'business_locations.default_payment_accounts',
                'business_locations.invoice_scheme_id',
                'business_locations.invoice_layout_id',
                'business_locations.customer_screen_picture',
                'tax_rates.amount',
                'business_locations.promotion_slider_speed',
                'business_locations.required_signature',
                'business_locations.show_numpad',
            );
        }

        $result = $query->get();        

        $locations = $result->pluck('name', 'id');

        $price_groups = SellingPriceGroup::forDropdown($business_id);

        if ($show_all) {
            $locations->prepend(__('report.all_locations'), '');
        }
        if ($receipt_printer_type_attribute) {
            $attributes = collect($result)->mapWithKeys(function ($item) use ($price_groups) {
                $default_payment_accounts = json_decode($item->default_payment_accounts, true);
                $default_payment_accounts['advance'] = [
                    'is_enabled' => 1,
                    'account' => null,
                ];                

                return [$item->id => [
                    'data-receipt_printer_type' => $item->receipt_printer_type,
                    'data-default_price_group' => ! empty($item->selling_price_group_id) && array_key_exists($item->selling_price_group_id, $price_groups) ? $item->selling_price_group_id : null,
                    'data-default_payment_accounts' => json_encode($default_payment_accounts),
                    'data-default_invoice_scheme_id' => $item->invoice_scheme_id,
                    'data-default_invoice_layout_id' => $item->invoice_layout_id,
                    'data-default_picture' => $item->customer_screen_picture,
                    'data-default_tax_amount' => $item->amount,
                    'data-promotion_slider_speed' => $item->promotion_slider_speed,
                    'data-required_signature' => $item->required_signature,
                    'data-show_numpad' => $item->show_numpad,
                ],
                ];
            })->all();

            return ['locations' => $locations, 'attributes' => $attributes];
        } else {
            return $locations;
        }
    }

    public static function forDropdownCashRegister($business_id, $cash_register_list_id, $location_id)
    {
        return CashRegisterList::where('business_id', $business_id)->where('location_id', $location_id)->pluck('name', 'id');
    }

    public function price_group()
    {
        return $this->belongsTo(\App\SellingPriceGroup::class, 'selling_price_group_id');
    }

    /**
     * Scope a query to only include active location.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', 1);
    }

    /**
     * Get the featured products.
     *
     * @return array/object
     */
    public function getFeaturedProducts($is_array = false, $check_location = true, $category = null, $display_option = null, $sub_category = null)
    {
                
        if($display_option == 'none') {
            $query = Variation::limit(0);
            $check_location = false;            
        } else if($display_option == 'all') {
            $query = Variation::join('product_locations as pl', 'pl.product_id', '=', 'variations.product_id')
            ->join('products as p', 'p.id', '=', 'variations.product_id')
            ->join('product_variations AS pv', 'variations.product_variation_id', '=', 'pv.id')
            ->where('p.not_for_selling', 0)
            ->with(['product_variation', 'product', 'media'])
                    ->select('variations.*',  DB::raw("IF(pv.is_dummy = 0, CONCAT(p.name, 
                    ' (', pv.name, ':',variations.name, ')'), p.name) AS product_name"));
        } else {   
            if (empty($this->featured_products)) {
                return [];
            } else {
                $featured_product = $this->featured_products;
                if(!is_array($this->featured_products)) {
                    $featured_product = str_replace(['\\', "[", "]", "\""], ['', '', '', ''], $this->featured_products);
                    $featured_product = explode(',', $featured_product);            
                }                       
            }         
            $query = Variation::whereIn('variations.id', $featured_product)
                    ->join('product_locations as pl', 'pl.product_id', '=', 'variations.product_id')
                    ->join('products as p', 'p.id', '=', 'variations.product_id')
                    ->join('product_variations AS pv', 'variations.product_variation_id', '=', 'pv.id')
                    ->where('p.not_for_selling', 0)
                    ->with(['product_variation', 'product', 'media'])
                    ->select('variations.*',  DB::raw("IF(pv.is_dummy = 0, CONCAT(p.name, 
                    ' (', pv.name, ':',variations.name, ')'), p.name) AS product_name"));
        }

        if ($check_location) {
            $query->where('pl.location_id', $this->id);
        }
        if ($category && ($category != 'all' && $category != 'null')) {    
            $category = explode(',', $category);        
            $query->whereIn('p.category_id', $category);
        }
        if ($sub_category && ($sub_category != 'all' && $sub_category != 'null')) {            
            $sub_category = explode(',', $sub_category);  
            if($category == 'null' || $category == null || !$category) {
                $query->whereIn('p.sub_category_id', $sub_category);
            } else {
                $query->orWhereIn('p.sub_category_id', $sub_category);
            }   
        }
        $featured_products = $query->get();        
         
        if ($is_array) {
            $array = [];
            foreach ($featured_products as $featured_product) {
                $array[$featured_product->id] = $featured_product->full_name;
            }

            return $array;
        }

        return $featured_products;
    }

    public function getLocationAddressAttribute()
    {
        $location = $this;
        $address_line_1 = [];
        if (! empty($location->landmark)) {
            $address_line_1[] = $location->landmark;
        }
        if (! empty($location->city)) {
            $address_line_1[] = $location->city;
        }
        if (! empty($location->state)) {
            $address_line_1[] = $location->state;
        }
        if (! empty($location->zip_code)) {
            $address_line_1[] = $location->zip_code;
        }
        $address = implode(', ', $address_line_1);
        $address_line_2 = [];
        if (! empty($location->country)) {
            $address_line_2[] = $location->country;
        }
        $address .= '<br>';
        $address .= implode(', ', $address_line_2);

        return $address;
    }
}
