<?php

namespace App\Http\Controllers\Auth;

use App\Business;
use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use Carbon\Carbon;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * All Utils instance.
     */
    protected $businessUtil;

    protected $moduleUtil;

    private $session_lifetime;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(BusinessUtil $businessUtil, ModuleUtil $moduleUtil)
    {
        $this->middleware('guest')->except('logout');
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;
    }

    public function showLoginForm()
    {        
        $business_id = session('business_id');        
        if($business_id) {
            $business = Business::find(session()->get('business_id'));            
            $lifetime = $business->session_lifetime ? $business->session_lifetime : 131400;
            config(['session.lifetime' => $lifetime]);
            env('SESSION_LIFETIME', $lifetime);
        }
        // dd(session());
        return view('auth.login');
    }

    /**
     * Change authentication from email to username
     *
     * @return void
     */
    public function username()
    {
        return 'username';
    }

    public function logout()
    {
        $this->businessUtil->activityLog(auth()->user(), 'logout');
        //Check if session has business id
        $business_id = session()->has('business') ? session('business.id') : null;
        Session::getHandler()->destroy(Session::getId());
        request()->session()->flush();
        \Auth::logout();
        request()->session()->regenerate(); 
        session()->put('business_id', $business_id);   

        return redirect('/login')->with(['session_lifetime' => $this->session_lifetime]);
    }

    /**
     * The user has been authenticated.
     * Check if the business is active or not.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  mixed  $user
     * @return mixed
     */
    protected function authenticated(Request $request, $user)
    {
        $this->businessUtil->activityLog($user, 'login', null, [], false, $user->business_id);
        $business = Business::find($user->business_id);
        
        if (! $user->business->is_active) {
            \Auth::logout();

            return redirect('/login')
              ->with(
                  'error',
                  ['success' => 0, 'msg' => __('lang_v1.business_inactive')]
              );
        } elseif ($user->status != 'active') {
            \Auth::logout();

            return redirect('/login')
              ->with(
                  'status',
                  ['success' => 0, 'msg' => __('lang_v1.user_inactive')]
              );
        } elseif (! $user->allow_login) {
            \Auth::logout();

            return redirect('/login')
                ->with(
                    'status',
                    ['success' => 0, 'msg' => __('lang_v1.login_not_allowed')]
                );
        } elseif (($user->user_type == 'user_customer') && ! $this->moduleUtil->hasThePermissionInSubscription($user->business_id, 'crm_module')) {
            \Auth::logout();

            return redirect('/login')
                ->with(
                    'status',
                    ['success' => 0, 'msg' => __('lang_v1.business_dont_have_crm_subscription')]
                );
        } 
        else if($business->working_hours_enable == 1) {
            $role = DB::table('model_has_roles')->where('model_id', $user->id)->first();
            $mytime = Carbon::now();
            $now = date('H:i:s', strtotime(Carbon::now()));      
            
                   
            if(in_array($role->role_id, json_decode($business->apply_to_working_hours))) {
                if((strtotime($now) < strtotime($business->start_hour_working) && strtotime($now) < strtotime($business->end_hour_working)) || strtotime($now) > strtotime($business->start_hour_working) && strtotime($now) > strtotime($business->end_hour_working)) {
                    \Auth::logout();
    
                    return redirect('/login')
                        ->withErrors(
                            [
                                'username' => __('lang_v1.working_hour_authenticate')
                            ]                        
                        );
                }            
            }
        }
    }

    protected function redirectTo()
    {
        
        $user = \Auth::user();
        if (! $user->can('dashboard.data') && $user->can('sell.create')) {
            return '/pos/create';
        }

        if ($user->user_type == 'user_customer') {
            return 'contact/contact-dashboard';
        }

        return '/home';
    }
}
