<?php

namespace App\Http\Controllers;

use App\Business;
use App\BusinessLocation;
use App\CashRegister;
use App\CashRegisterList;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class CashRegisterListController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // start
        if (! auth()->user()->can('cash_register_list.view') && ! auth()->user()->can('cash_register_list.create')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');

            $cashRegisterList = CashRegisterList::where('cash_registers_list.business_id', $business_id)
                        ->select(['cash_registers_list.name as cash_regsiter_name', 'business.name as business_name', 'business_locations.name as location_name', 'cash_registers_list.id'])
                        ->leftJoin('business', 'business.id', '=', 'cash_registers_list.business_id')
                        ->leftJoin('business_locations', 'business_locations.id', '=', 'cash_registers_list.location_id')->get();            

            return DataTables::of($cashRegisterList)
                ->addColumn(
                    'action',
                    '@can("brand.update")
                    <button data-href="{{action(\'App\Http\Controllers\CashRegisterListController@edit\', [$id])}}" class="btn btn-xs btn-primary edit_cash_register_list_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                        &nbsp;
                    @endcan
                    @can("brand.delete")
                    <button data-href="{{action(\'App\Http\Controllers\CashRegisterListController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_cash_register_list_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>
                    @endcan'
                )              
                ->removeColumn('id')  
                ->rawColumns([3])
                ->make(false);
        }

        return view('cash_register_list.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
        if (! auth()->user()->can('cash_register_list.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = Business::where('id', $business_id)->get()->pluck('name', 'id');
        
        $business_location = BusinessLocation::where('business_id', $business_id)->get()->pluck('name', 'id');

        return view('cash_register_list.create')
                ->with(compact('business', 'business_location'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        if (! auth()->user()->can('cash_register.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $input = $request->only(['name', 'business_id', 'location_id']);
            $input['created_by'] = $request->session()->get('user.id');

            $cash_register = CashRegisterList::create($input);
            $output = ['success' => true,
                'data' => $cash_register,
                'msg' => __('cash_register.added_success'),
            ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
        if(! auth()->user()->can('cash_register_list.update')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $business = Business::where('id', $business_id)->get()->pluck('name', 'id');
        
            $business_location = BusinessLocation::where('business_id', $business_id)->get()->pluck('name', 'id');
            $crl = CashRegisterList::where('business_id', $business_id)->find($id);            

            return view('cash_register_list.edit')
                ->with(compact('crl', 'business', 'business_location'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
        if (! auth()->user()->can('cash_register.udpate')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $input = $request->only(['name', 'business_id', 'location_id']);
            $input['created_by'] = $request->session()->get('user.id');
            $business_id = request()->session()->get('user.business_id');
            $cash_register = CashRegisterList::where('business_id', $business_id)->findOrFail($id);
            $cash_register->name = $input['name'];
            $cash_register->business_id = $input['business_id'];
            $cash_register->location_id = $input['location_id'];
            $cash_register->save();
            $output = ['success' => true,
                'data' => $cash_register,
                'msg' => __('cash_register.updated_success'),
            ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            
            $output = ['success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
        if (! auth()->user()->can('cash_register.delete')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $business_id = request()->user()->business_id;

                $cash_register = CashRegisterList::where('business_id', $business_id)->findOrFail($id);
                $cash_register->delete();

                $output = ['success' => true,
                    'msg' => __('cash_register.deleted_success'),
                ];
            } catch (\Exception $e) {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }
}
