<?php

namespace App\Http\Controllers;

use App\Barcode;
use App\Product;
use App\SellingPriceGroup;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LabelsController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $transactionUtil;

    protected $productUtil;

    /**
     * Constructor
     *
     * @param  TransactionUtil  $TransactionUtil
     * @return void
     */
    public function __construct(TransactionUtil $transactionUtil, ProductUtil $productUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $productUtil;
    }

    /**
     * Display labels
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        if (! auth()->user()->can('product.print_labels')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = $request->session()->get('user.business_id');
        $purchase_id = $request->get('purchase_id', false);
        $product_id = $request->get('product_id', false);

        //Get products for the business
        $products = [];
        $price_groups = [];
        if ($purchase_id) {
            $products = $this->transactionUtil->getPurchaseProducts($business_id, $purchase_id);
        } elseif ($product_id) {
            $products = $this->productUtil->getDetailsFromProduct($business_id, $product_id);
        }

        //get price groups
        $price_groups = [];
        if (! empty($purchase_id) || ! empty($product_id)) {
            $price_groups = SellingPriceGroup::where('business_id', $business_id)
                                    ->active()
                                    ->pluck('name', 'id');
        }

        $barcode_settings = Barcode::where('business_id', $business_id)
                                ->orWhereNull('business_id')
                                ->select(DB::raw('CONCAT(name, ", ", COALESCE(description, "")) as name, id, is_default'))
                                ->get();
        $default = $barcode_settings->where('is_default', 1)->first();
        $barcode_settings = $barcode_settings->pluck('name', 'id');
        $information_labels = DB::table('information_labels')->get()->toArray();

        return view('labels.show')
            ->with(compact('products', 'barcode_settings', 'default', 'price_groups', 'information_labels'));
    }

    /**
     * Returns the html for product row
     *
     * @return \Illuminate\Http\Response
     */
    public function addProductRow(Request $request)
    {
        if ($request->ajax()) {
            $product_id = $request->input('product_id');
            $variation_id = $request->input('variation_id');
            $business_id = $request->session()->get('user.business_id');

            if (! empty($product_id)) {
                $index = $request->input('row_count');
                $products = $this->productUtil->getDetailsFromProduct($business_id, $product_id, $variation_id);

                $price_groups = SellingPriceGroup::where('business_id', $business_id)
                                            ->active()
                                            ->pluck('name', 'id');

                return view('labels.partials.show_table_rows')
                        ->with(compact('products', 'index', 'price_groups'));
            }
        }
    }

    /**
     * Returns the html for labels preview
     *
     * @return \Illuminate\Http\Response
     */
    public function preview(Request $request)
    {
        try {
            $products = $request->get('products');
            $print = $request->get('print');
            $barcode_setting = $request->get('barcode_setting');
            $business_id = $request->session()->get('user.business_id');

            $barcode_details = Barcode::find($barcode_setting);
            $barcode_details->stickers_in_one_sheet = $barcode_details->is_continuous ? $barcode_details->stickers_in_one_row : $barcode_details->stickers_in_one_sheet;
            $barcode_details->paper_height = $barcode_details->is_continuous ? $barcode_details->height : $barcode_details->paper_height;
            if ($barcode_details->stickers_in_one_row == 1) {
                $barcode_details->col_distance = 0;
                $barcode_details->row_distance = 0;
            }
            // if($barcode_details->is_continuous){
            //     $barcode_details->row_distance = 0;
            // }

            $business_name = $request->session()->get('business.name');

            $product_details_page_wise = [];
            $total_qty = 0;
            foreach ($products as $value) {
                $details = $this->productUtil->getDetailsFromVariation($value['variation_id'], $business_id, null, false);

                if (! empty($value['exp_date'])) {
                    $details->exp_date = $value['exp_date'];
                }
                if (! empty($value['packing_date'])) {
                    $details->packing_date = $value['packing_date'];
                }
                if (! empty($value['lot_number'])) {
                    $details->lot_number = $value['lot_number'];
                }

                if (! empty($value['price_group_id'])) {
                    $tax_id = $print['price_type'] == 'inclusive' ?: $details->tax_id;

                    $group_prices = $this->productUtil->getVariationGroupPrice($value['variation_id'], $value['price_group_id'], $tax_id);

                    $details->sell_price_inc_tax = $group_prices['price_inc_tax'];
                    $details->default_sell_price = $group_prices['price_exc_tax'];
                }

                for ($i = 0; $i < $value['quantity']; $i++) {
                    $page = intdiv($total_qty, $barcode_details->stickers_in_one_sheet);

                    if ($total_qty % $barcode_details->stickers_in_one_sheet == 0) {
                        $product_details_page_wise[$page] = [];
                    }

                    $product_details_page_wise[$page][] = $details;
                    $total_qty++;
                }
            }

            $margin_top = $barcode_details->is_continuous ? 0 : $barcode_details->top_margin * 1;
            $margin_left = $barcode_details->is_continuous ? 0 : $barcode_details->left_margin * 1;
            $paper_width = $barcode_details->paper_width * 1;
            $paper_height = $barcode_details->paper_height * 1;

            // print_r($paper_height);
            // echo "==";
            // print_r($margin_left);exit;

            // $mpdf = new \Mpdf\Mpdf(['mode' => 'utf-8',
            //             'format' => [$paper_width, $paper_height],
            //             'margin_top' => $margin_top,
            //             'margin_bottom' => $margin_top,
            //             'margin_left' => $margin_left,
            //             'margin_right' => $margin_left,
            //             'autoScriptToLang' => true,
            //             // 'disablePrintCSS' => true,
            // 'autoLangToFont' => true,
            // 'autoVietnamese' => true,
            // 'autoArabic' => true
            //             ]
            //         );
            //print_r($mpdf);exit;

            $i = 0;
            $len = count($product_details_page_wise);
            $is_first = false;
            $is_last = false;

            //$original_aspect_ratio = 4;//(w/h)
            $factor = (($barcode_details->width / $barcode_details->height)) / ($barcode_details->is_continuous ? 2 : 4);
            $html = '';
            foreach ($product_details_page_wise as $page => $page_products) {
                if ($i == 0) {
                    $is_first = true;
                }

                if ($i == $len - 1) {
                    $is_last = true;
                }

                $output = view('labels.partials.preview_2')
                            ->with(compact('print', 'page_products', 'business_name', 'barcode_details', 'margin_top', 'margin_left', 'paper_width', 'paper_height', 'is_first', 'is_last', 'factor'))->render();
                print_r($output);
                //$mpdf->WriteHTML($output);

                // if($i < $len - 1){
                //     // '', '', '', '', '', '', $margin_left, $margin_left, $margin_top, $margin_top, '', '', '', '', '', '', 0, 0, 0, 0, '', [$barcode_details->paper_width*1, $barcode_details->paper_height*1]
                //     $mpdf->AddPage();
                // }

                $i++;
            }

            print_r('<script>window.print()</script>');
            exit;
            //return $output;

            //$mpdf->Output();

            // $page_height = null;
            // if ($barcode_details->is_continuous) {
            //     $rows = ceil($total_qty/$barcode_details->stickers_in_one_row) + 0.4;
            //     $barcode_details->paper_height = $barcode_details->top_margin + ($rows*$barcode_details->height) + ($rows*$barcode_details->row_distance);
            // }

            // $output = view('labels.partials.preview')
            //     ->with(compact('print', 'product_details', 'business_name', 'barcode_details', 'product_details_page_wise'))->render();

            // $output = ['html' => $html,
            //                 'success' => true,
            //                 'msg' => ''
            //             ];
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = __('lang_v1.barcode_label_error');
        }

        //return $output;
    }

    public function updateSettings(Request $request)
    {
        try {
            //code...                        
            foreach ($request->all() as $key => $value) {
                # code...
                if($key != 'barcode_setting') {
                    $check =  DB::table('information_labels')->where('name', $key)->first();
                    if($check == null) {
                        DB::table('information_labels')->where('name', $key)->insert([
                            'name' => $key,
                            'size' => $value
                        ]);                     
                    } else {
                        DB::table('information_labels')->where('name', $key)->update([
                            'size' => $value
                        ]); 
                    }
                } else {
                    DB::table('barcodes')->update(['is_default' => 0]);
                    $barcode = Barcode::find($value);
                    $barcode->is_default = 1;
                    $barcode->save();
                }
            }             

            return response()->json([
                'rc' => 1,
                'rm' => 'Settings Size Success'
            ]); 
        } catch (\Exception $e) {
            //throw $th;            
            return response()->json([
                'rc' => 500,
                'rm' => 'Something went wrong'
            ]);
        }
    }

    public function addToPrintLabelQueue(Request $request)
    {
        try {
            //code...                        
            $check_total_data = DB::table('print_labels_queue')->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->count();                        
            if($check_total_data >= 15) {                
                return response()->json([
                    'rc' => 2,
                    'rm' => 'You have reached the print queue limit',
                ]); 
            } else {
                $check_reference_id = DB::table('print_labels_queue')->where('reference_id', $request->reference_id)->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->exists();                
                if($check_reference_id) {
                    return response()->json([
                        'rc' => 2,
                        'rm' => 'Product already added to queue',
                    ]); 
                } else {
                    DB::table('print_labels_queue')->insert([
                        'reference_id' => $request->reference_id,
                        'reference_table' => $request->reference_table,
                        'user_id' => auth()->user()->id,
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                    $total_queue = DB::table('print_labels_queue')->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->count();
                    return response()->json([
                        'rc' => 1,
                        'rm' => 'Add To Print Queue Successfully',
                        'total_queue' => $total_queue
                    ]); 
                }
            }

        } catch (\Exception $e) {
            //throw $th;                        
            return response()->json([
                'rc' => 500,
                'rm' => 'Something went wrong'
            ]);
        }
    }

    public function deleteItemPrintLabelQueue(Request $request)
    {
        try {
            //code...
            DB::beginTransaction();
            DB::table('print_labels_queue')->where('reference_id', $request->reference_id)->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->delete();
            $total_queue = DB::table('print_labels_queue')->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->count();
            DB::commit();
            return response()->json([
                'rc' => 1,
                'rm' => 'Delete Item Print Queue Successfully',
                'total_queue' => $total_queue
            ]); 
        } catch (\Exception $e) {
            //throw $th;
            return response()->json([
                'rc' => 500,
                'rm' => 'Something went wrong'
            ]);
        }
    }

    public function popupPrintLabelsQueue(Request $request)
    {
        try {
            //code...            
            $seller_id = $request->seller_id;
            $reference_table = $request->reference_table;

            $print_labels_queue = DB::table('print_labels_queue')->where('reference_table', $reference_table)->where('user_id', auth()->user()->id)->get()->toArray();

            $data = [];
            if(!empty($print_labels_queue)) {
                foreach ($print_labels_queue as $plq) {
                    # code...                    
                    $amazon_data = DB::table($plq->reference_table)->where('id', $plq->reference_id)->first();
                    array_push($data, $amazon_data);
                }
            }            

            return view('labels.partials.popup_print_labels_queue')->with(compact(
                'data',
                'seller_id',
            ));
        } catch (\Exception $e) {
            //throw $th;
            // Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());
            return response()->json('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());
        }
    }

    public function getPrintQueueCountByUser(Request $request)
    {
        $total_queue = DB::table('print_labels_queue')->where('reference_table', $request->reference_table)->where('user_id', auth()->user()->id)->count();

        return $total_queue;
    }
}
