<?php

namespace App\Services;
use Aws\Credentials\Credentials;
use Aws\Signature\SignatureV4;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class AmazonSpApiService
{
    protected $config;
    protected $endpoints = [
        'ATVPDKIKX0DER' => [ // US marketplace
            'endpoint' => 'https://sandbox.sellingpartnerapi-na.amazon.com',
            'region' => 'us-east-1'
        ],
        'A2EUQ1WTGCTBG2' => [ // CA marketplace
            'endpoint' => 'https://sandbox.sellingpartnerapi-na.amazon.com',
            'region' => 'us-east-1'
        ],
        'A1F83G8C2ARO7P' => [ // UK marketplace
            'endpoint' => 'https://sandbox.sellingpartnerapi-eu.amazon.com',
            'region' => 'eu-west-1'
        ]
    ];

    public function __construct()
    {
        $this->config = config('amazon');
        $this->validateMarketplace();
    }

    /**
     * Validate marketplace ID and configuration
     */
    private function validateMarketplace()
    {
        $marketplaceId = $this->config['marketplace_id'];
        
        if (!isset($this->endpoints[$marketplaceId])) {
            throw new \Exception("Invalid or unsupported MarketplaceId: {$marketplaceId}");
        }
    }

    /**
     * Get endpoint configuration for current marketplace
     */
    private function getEndpointConfig()
    {
        return $this->endpoints[$this->config['marketplace_id']];
    }

    /**
     * Get LWA Access Token from Amazon SP-API
     */
    private function getAccessToken()
    {
        //$url = 'https://api.amazon.com/auth/o2/token';
        $url = 'https://api.sandbox.amazon.com/auth/o2/token';
        $postData = [
            'grant_type'    => 'refresh_token',
            'refresh_token' => $this->config['refresh_token'],
            'client_id'     => $this->config['lwa_client_id'],
            'client_secret' => $this->config['lwa_client_secret'],
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_VERBOSE, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        $result = json_decode($response, true);

        if ($httpCode !== 200) {
            throw new \Exception(
                'Failed to get LWA Access Token. HTTP Code: ' . $httpCode . 
                ', Error: ' . ($result['error_description'] ?? $curlError ?? 'Unknown error') .
                ', Response: ' . $response
            );
        }

        if (empty($result['access_token'])) {
            throw new \Exception(
                'Invalid LWA response format. Response: ' . $response
            );
        }

        return $result['access_token'];
    }

    

    /**
     * Get Amazon Orders using cURL
     */
    public function getOrders()
    {
        $accessToken = $this->getAccessToken(); // LWA Token from Amazon
        $endpointConfig = $this->getEndpointConfig();
        $region = $endpointConfig['region'];
        $service = 'execute-api';

        // $url = 'https://sellingpartnerapi-na.amazon.com/orders/v0/orders?' . http_build_query([
        //     'MarketplaceIds' => $this->config['marketplace_id'],
        //     'CreatedAfter'   => date('c', strtotime('-7 days'))
        // ]);

        // Set CreatedAfter to 7 days ago and format according to ISO 8601
        $createdAfter = gmdate('Y-m-d\TH:i:s\Z', strtotime('-7 days'));
        
        // Build the base URL using the correct endpoint for the marketplace
        $baseUrl = $endpointConfig['endpoint'] . '/orders/v0/orders';
        
        // Build query parameters manually to handle repeated parameters correctly
        $queryParts = [];
        
        // Add basic parameters
        $queryParts[] = 'CreatedAfter=' . urlencode($createdAfter);
        $queryParts[] = 'MaxResultsPerPage=100';
        
        // Add MarketplaceIds (can be repeated for multiple marketplaces)
        $queryParts[] = 'marketplaceids=' . urlencode($this->config['marketplace_id']);
        
        
        // Add OrderStatuses as repeated parameters
        // foreach (['UNSHIPPED', 'PARTIALLY_SHIPPED', 'SHIPPED'] as $status) {
        //     $queryParts[] = 'OrderStatuses=' . urlencode($status);
        // }
        $queryParts[] = 'OrderStatuses=' . urlencode('UNSHIPPED,PARTIALLY_SHIPPED,SHIPPED');
        
        // Build the final URL
        $url = $baseUrl . '?' . implode('&', $queryParts);

        $credentials = new Credentials(
            $this->config['aws_access_key_id'],
            $this->config['aws_secret_access_key']
        );
        $signer = new SignatureV4($service, $region);

        $headers = [
            'x-amz-access-token' => $accessToken,
            'Accept' => 'application/json',
            'Host' => parse_url($url, PHP_URL_HOST),
            'x-amz-date' => gmdate('Ymd\THis\Z')
        ];

        try {
            $request = new \GuzzleHttp\Psr7\Request('GET', $url, $headers);
            $signedRequest = $signer->signRequest($request, $credentials);

            // Send request using CURL with SSL verification
            $ch = curl_init((string)$signedRequest->getUri());
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $this->formatHeaders($signedRequest->getHeaders()));
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_VERBOSE, true);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
        } catch (\Exception $e) {
            return [
                'error' => 'AWS Request Signing Error',
                'response' => $e->getMessage(),
                'debug' => [
                    'url' => $url,
                    'region' => $region,
                    'marketplace_id' => $this->config['marketplace_id']
                ]
            ];
        }

        // Debug information
        $debug = [
            'url' => $url,
            'headers' => $this->formatHeaders($signedRequest->getHeaders()),
            'curl_error' => $curlError,
            'http_code' => $httpCode,
            'response' => $response
        ];

        if ($httpCode !== 200) {
            return [
                'error' => "Failed to fetch orders. HTTP Code: $httpCode",
                'response' => $response,
                'debug' => $debug
            ];
        }

        return json_decode($response, true);
    }

    /**
     * Get Amazon Products using Catalog API
     */
    public function getProducts()
    {
        $accessToken = $this->getAccessToken(); // LWA Token from Amazon
        $endpointConfig = $this->getEndpointConfig();
        $region = $endpointConfig['region'];
        $service = 'execute-api';

        // Use FBA Inventory API endpoint
        $baseUrl = $endpointConfig['endpoint'] . '/fba/inventory/v1/inventorySummaries';
        
        // Build query parameters
        $queryParts = [];
        
        // Add required parameters
        $queryParts[] = 'details=false';
        $queryParts[] = 'startDateTime=' . urlencode(gmdate('Y-m-d\TH:i:s\Z', strtotime('-30 days')));
        $queryParts[] = 'sellerSkus=' . urlencode('TEST-SKU-001'); // Add a test SKU
        
        // Build the final URL
        $url = $baseUrl . '?' . implode('&', $queryParts);

        $credentials = new Credentials(
            $this->config['aws_access_key_id'],
            $this->config['aws_secret_access_key']
        );
        $signer = new SignatureV4($service, $region);

        $headers = [
            'x-amz-access-token' => $accessToken,
            'Accept' => 'application/json',
            'Host' => parse_url($url, PHP_URL_HOST),
            'x-amz-date' => gmdate('Ymd\THis\Z')
        ];

        try {
            $request = new \GuzzleHttp\Psr7\Request('GET', $url, $headers);
            $signedRequest = $signer->signRequest($request, $credentials);

            // Send request using CURL with SSL verification
            $ch = curl_init((string)$signedRequest->getUri());
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $this->formatHeaders($signedRequest->getHeaders()));
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_VERBOSE, true);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
        } catch (\Exception $e) {
            return [
                'error' => 'AWS Request Signing Error',
                'response' => $e->getMessage(),
                'debug' => [
                    'url' => $url,
                    'region' => $region,
                    'marketplace_id' => $this->config['marketplace_id']
                ]
            ];
        }

        // Debug information
        $debug = [
            'url' => $url,
            'headers' => $this->formatHeaders($signedRequest->getHeaders()),
            'curl_error' => $curlError,
            'http_code' => $httpCode,
            'response' => $response
        ];

        if ($httpCode !== 200) {
            return [
                'error' => "Failed to fetch products. HTTP Code: $httpCode",
                'response' => $response,
                'debug' => $debug
            ];
        }

        return json_decode($response, true);
    }

    /**
     * Get specific product details by ASIN
     */
    public function getProductByAsin($asin)
    {
        $accessToken = $this->getAccessToken(); // LWA Token from Amazon
        $endpointConfig = $this->getEndpointConfig();
        $region = $endpointConfig['region'];
        $service = 'execute-api';

        // Build the base URL using the correct endpoint for the marketplace
        $baseUrl = $endpointConfig['endpoint'] . '/catalog/2022-04-01/items/' . urlencode($asin);
        
        // Build query parameters
        $queryParts = [];
        
        // Add MarketplaceIds
        $queryParts[] = 'marketplaceids=' . urlencode($this->config['marketplace_id']);
        
        // Add optional parameters for detailed information
        $queryParts[] = 'includedData=' . urlencode('summaries,attributes,dimensions,identifiers,images,productTypes,relationships,salesRanks');
        
        // Build the final URL
        $url = $baseUrl . '?' . implode('&', $queryParts);

        $credentials = new Credentials(
            $this->config['aws_access_key_id'],
            $this->config['aws_secret_access_key']
        );
        $signer = new SignatureV4($service, $region);

        $headers = [
            'x-amz-access-token' => $accessToken,
            'Accept' => 'application/json',
            'Host' => parse_url($url, PHP_URL_HOST),
            'x-amz-date' => gmdate('Ymd\THis\Z')
        ];

        try {
            $request = new \GuzzleHttp\Psr7\Request('GET', $url, $headers);
            $signedRequest = $signer->signRequest($request, $credentials);

            // Send request using CURL with SSL verification
            $ch = curl_init((string)$signedRequest->getUri());
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $this->formatHeaders($signedRequest->getHeaders()));
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_VERBOSE, true);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
        } catch (\Exception $e) {
            return [
                'error' => 'AWS Request Signing Error',
                'response' => $e->getMessage(),
                'debug' => [
                    'url' => $url,
                    'region' => $region,
                    'marketplace_id' => $this->config['marketplace_id']
                ]
            ];
        }

        // Debug information
        $debug = [
            'url' => $url,
            'headers' => $this->formatHeaders($signedRequest->getHeaders()),
            'curl_error' => $curlError,
            'http_code' => $httpCode,
            'response' => $response
        ];

        if ($httpCode !== 200) {
            return [
                'error' => "Failed to fetch product details. HTTP Code: $httpCode",
                'response' => $response,
                'debug' => $debug
            ];
        }

        return json_decode($response, true);
    }

    /**
     * Search Amazon Products by keywords
     */
    public function searchProducts($keywords)
    {
        $accessToken = $this->getAccessToken();
        $endpointConfig = $this->getEndpointConfig();
        $region = $endpointConfig['region'];
        $service = 'execute-api';

        // Build the base URL using the correct endpoint for the marketplace
        $baseUrl = $endpointConfig['endpoint'] . '/catalog/2022-04-01/items/search';
        
        // Build query parameters
        $queryParts = [];
        
        // Add required parameters
        $queryParts[] = 'marketplaceIds=' . urlencode($this->config['marketplace_id']);
        $queryParts[] = 'keywords=' . urlencode($keywords);
        
        // Add optional parameters
        $queryParts[] = 'includedData=' . urlencode('summaries,attributes,dimensions,identifiers,images,productTypes,relationships,salesRanks');
        $queryParts[] = 'pageSize=20';
        
        // Build the final URL
        $url = $baseUrl . '?' . implode('&', $queryParts);

        $credentials = new Credentials(
            $this->config['aws_access_key_id'],
            $this->config['aws_secret_access_key']
        );
        $signer = new SignatureV4($service, $region);

        $headers = [
            'x-amz-access-token' => $accessToken,
            'Accept' => 'application/json',
            'Host' => parse_url($url, PHP_URL_HOST),
            'x-amz-date' => gmdate('Ymd\THis\Z')
        ];

        try {
            $request = new \GuzzleHttp\Psr7\Request('GET', $url, $headers);
            $signedRequest = $signer->signRequest($request, $credentials);

            // Send request using CURL with SSL verification
            $ch = curl_init((string)$signedRequest->getUri());
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $this->formatHeaders($signedRequest->getHeaders()));
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_VERBOSE, true);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
        } catch (\Exception $e) {
            return [
                'error' => 'AWS Request Signing Error',
                'response' => $e->getMessage(),
                'debug' => [
                    'url' => $url,
                    'region' => $region,
                    'marketplace_id' => $this->config['marketplace_id']
                ]
            ];
        }


        // Debug information
        $debug = [
            'url' => $url,
            'headers' => $this->formatHeaders($signedRequest->getHeaders()),
            'curl_error' => $curlError,
            'http_code' => $httpCode,
            'response' => $response
        ];

        if ($httpCode !== 200) {
            return [
                'error' => "Failed to search products. HTTP Code: $httpCode",
                'response' => $response,
                'debug' => $debug
            ];
        }

        return json_decode($response, true);
    }

    private function formatHeaders($headers)
    {
        $formatted = [];
        foreach ($headers as $name => $values) {
            $formatted[] = $name . ': ' . implode(', ', $values);
        }
        return $formatted;
    }

    function fetchProductDetails($asin = null) {
        $baseUrl = 'https://sandbox.sellingpartnerapi-na.amazon.com/catalog/v0/items';

        // Amazon requires comma-separated IDs, not array-style query
        $params = [
            'ASIN' => $asin,
            'MarketplaceIds' => 'ATVPDKIKX0DER' // Correct format
        ];

        // Manually create query to avoid [0] index issue
        $queryString = 'ASIN=' . urlencode($params['ASIN']) . 
                    '&MarketplaceIds=' . urlencode($params['MarketplaceIds']);

        // Construct URL
        $url = $baseUrl . '?' . $queryString;
        

        // Generate tokens and signature
        $accessToken = $this->getAccessToken(); 
        $signature = $this->generateSignature($url);

        // Headers
        $headers = [
            'x-amz-access-token' => $accessToken,
            'Accept' => 'application/json',
            'X-Amz-Date' => gmdate('Ymd\THis\Z'),
            'Authorization' => $signature,
        ];

        //dd($headers);

        // Perform request
        $response = Http::withHeaders($headers)->get($url);

        // View full response (for debugging)
        return $response->json();
    }

    function generateSignature($url) {
        // Your credentials and configurations
        $accessKey = $this->config['aws_access_key_id']; // Replace with your AWS access key
        $secretKey = $this->config['aws_secret_access_key']; // Replace with your AWS secret key
        $region = 'us-east-1'; // Your AWS region
        $service = 'execute-api'; // The API service
        $host = parse_url($url, PHP_URL_HOST);
        $date = Carbon::now()->format('Ymd');
        $time = Carbon::now()->format('Ymd\THis\Z');
        
        // Parse URL and build the canonical request
        $parsedUrl = parse_url($url);
        $uri = $parsedUrl['path'] ?? '/';
        $queryString = isset($parsedUrl['query']) ? $parsedUrl['query'] : '';
        
        // Sort query parameters for consistent signing
        parse_str($queryString, $queryParams);
        ksort($queryParams);
        $queryString = http_build_query($queryParams);
        $accessToken = $this->getAccessToken(); // LWA Token from Amazon
        
        // The canonical request
        $canonicalRequest = [
            'GET',                   // HTTP method
            $uri,                    // Request path
            $queryString,            // Query string
            'host:' . $host,         // Request headers
            'x-amz-access-token:' . $accessToken,
            'x-amz-date:' . $time,
            '',                      // Payload (empty for GET requests)
        ];
        
        // Create the canonical request string
        $canonicalRequestStr = implode("\n", $canonicalRequest);
        
        // Create the string to sign
        $stringToSign = [
            'AWS4-HMAC-SHA256',
            $time,
            $date . '/' . $region . '/' . $service . '/aws4_request',
            hash('sha256', $canonicalRequestStr)
        ];
        
        $stringToSignStr = implode("\n", $stringToSign);
        
        // Derive the signing key
        $kDate = hash_hmac('sha256', $date, 'AWS4' . $secretKey, true);
        $kRegion = hash_hmac('sha256', $region, $kDate, true);
        $kService = hash_hmac('sha256', $service, $kRegion, true);
        $kSigning = hash_hmac('sha256', 'aws4_request', $kService, true);
        
        // Final signature
        $signature = hash_hmac('sha256', $stringToSignStr, $kSigning);
        
        // Return the signature
        return 'AWS4-HMAC-SHA256 Credential=' . $accessKey . '/' . $date . '/' . $region . '/' . $service . '/aws4_request, SignedHeaders=host;x-amz-access-token;x-amz-date, Signature=' . $signature;
    }

}
