<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\CashRegister;
use App\CashRegisterList;
use App\Utils\CashRegisterUtil;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;

class CashRegisterController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $cashRegisterUtil;

    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param  CashRegisterUtil  $cashRegisterUtil
     * @return void
     */
    public function __construct(CashRegisterUtil $cashRegisterUtil, ModuleUtil $moduleUtil)
    {
        $this->cashRegisterUtil = $cashRegisterUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('cash_register.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request, $message = null)
    {
        //like:repair                                
        $sub_type = request()->get('sub_type');
        $cash_register_list_id = $request->get('cash_register_list_id');        
        $location_id = $request->get('location_id');        
        // Check if there is a open register, if yes then redirect to POS screen.
        if($cash_register_list_id) {
            $c = $this->cashRegisterUtil->checkOpenedRegister($cash_register_list_id);            
            if($c['count'] > 0) {
                return redirect()->action([\App\Http\Controllers\SellPosController::class, 'create'], ['sub_type' => $sub_type]);
            }
        } else {
            if ($this->cashRegisterUtil->countOpenedRegister() != 0) {
                return redirect()->action([\App\Http\Controllers\SellPosController::class, 'create'], ['sub_type' => $sub_type]);
            }
        }
        $business_id = request()->session()->get('user.business_id');
        if($cash_register_list_id) {
            $business_locations = BusinessLocation::forDropdown($business_id);
            $cash_registers_list = BusinessLocation::forDropdownCashRegister($business_id, $cash_register_list_id, $location_id);
            // array_splice($cash_register_list_id,0, 0, array(null => 'Please Select'));
        } else {
            $business_locations = BusinessLocation::forDropdown($business_id);
            $cash_registers_list = [];
        }        

        $pos_settings = ! empty(request()->session()->get('business.pos_settings')) ? json_decode(request()->session()->get('business.pos_settings'), true) : [];

        // dd($business_locations);

        return view('cash_register.create')->with(compact('business_locations', 'sub_type', 'cash_register_list_id', 'location_id', 'cash_registers_list', 'pos_settings'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //like:repair
        // dd($request->all());
        $sub_type = request()->get('sub_type');
        $checkOpenedByUser = $this->cashRegisterUtil->checkOpenedRegister($request->input('cash_register_list_id'));    
        if(!$request->location_id) {
            return redirect()->back()->withInput($request->all())->with('errors', 'Location should be selected');
        }    
        if(!$request->cash_register_list_id) {
            return redirect()->back()->withInput($request->all())->with('errors', 'Cash Register Name should be selected');
        }    
        if($checkOpenedByUser['count'] > 0) {
            if(auth()->user()->id != $checkOpenedByUser['user_id']) {                                                            
                return redirect()->back()->withInput($request->all())->with('errors', 'Cash Register Already Opened By Another User');
            }
        }            
        try {

            $initial_amount = 0;
            if (! empty($request->input('amount'))) {
                $initial_amount = $this->cashRegisterUtil->num_uf($request->input('amount'));
            }
            $user_id = $request->session()->get('user.id');
            $business_id = $request->session()->get('user.business_id');

            $register = CashRegister::create([
                'business_id' => $business_id,
                'user_id' => $user_id,
                'status' => 'open',
                'location_id' => $request->input('location_id'),
                'created_at' => \Carbon::now('GMT')->format('Y-m-d H:i:00'),
                'cash_register_list_id' => $request->input('cash_register_list_id'),
            ]);
            if (! empty($initial_amount)) {
                $register->cash_register_transactions()->create([
                    'amount' => $initial_amount,
                    'pay_method' => 'cash',
                    'type' => 'credit',
                    'transaction_type' => 'initial',
                    'cash_register_list_id' => $request->input('cash_register_list_id'),
                ]);
            }

            // action([\App\Http\Controllers\SellController::class, 'saveCashDrawerOpenings'], ['reason' => $request->input('reason_cash_drawer_')]);
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
        }

        return redirect()->action([\App\Http\Controllers\SellPosController::class, 'create'], ['sub_type' => $sub_type]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\CashRegister  $cashRegister
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (! auth()->user()->can('view_cash_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');        
        $register_details = $this->cashRegisterUtil->getRegisterDetails($id);        
        // dd($register_details)
        $user_id = $register_details->user_id;
        $open_time = $register_details['open_time'];
        $close_time = ! empty($register_details['closed_at']) ? $register_details['closed_at'] : \Carbon::now('GMT')->toDateTimeString();
        // dd($open_time, $close_time);
        $details = $this->cashRegisterUtil->getRegisterTransactionDetails($user_id, $register_details->cash_register_list_id, $open_time, $close_time);        

        $payment_types = $this->cashRegisterUtil->payment_types(null, false, $business_id);

        return view('cash_register.register_details')
                    ->with(compact('register_details', 'details', 'payment_types', 'close_time'));
    }

    /**
     * Shows register details modal.
     *
     * @param  void
     * @return \Illuminate\Http\Response
     */
    public function getRegisterDetails(Request $request)
    {        
        if (! auth()->user()->can('view_cash_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $cash_register_list_id = $request->get('cash_register_list_id');        

        $register_details = $this->cashRegisterUtil->getRegisterDetails($request->get('cash_register_list_id'));        
        
        $user_id = $register_details ? $register_details->user_id : auth()->user()->id;
        $open_time = $register_details['open_time'];
        // $close_time = \Carbon::now('GMT')->toDateTimeString();
        $close_time = $register_details['closed_at'] ? $register_details['closed_at'] : \Carbon::now('GMT')->toDateTimeString();
        // dd($open_time, $close_time);
        
        $is_types_of_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');
                
        $sell_no_registers = $this->cashRegisterUtil->getNoRegisterDetails($open_time, $close_time);
        
        $details = $this->cashRegisterUtil->getRegisterTransactionDetails($user_id, $cash_register_list_id, $open_time, $close_time, $is_types_of_service_enabled);
        
        $payment_types = $this->cashRegisterUtil->payment_types($register_details->location_id, true, $business_id);
        
        return view('cash_register.register_details')
                ->with(compact('register_details', 'details', 'payment_types', 'close_time', 'sell_no_registers'));
    }

    /**
     * Shows close register form.
     *
     * @param  void
     * @return \Illuminate\Http\Response
     */
    public function getCloseRegister($id = null)
    {        
        if (! auth()->user()->can('close_cash_register')) {
            abort(403, 'Unauthorized action.');
        }

        // var_dump(request()->segment(1));
        // dd('reset');
        $business_id = request()->session()->get('user.business_id');
        $register_details = $this->cashRegisterUtil->getRegisterDetails($id);

        $user_id = $register_details ? $register_details->user_id : auth()->user()->id;
        $open_time = $register_details['open_time'];
        $close_time = $register_details['closed_at'] ? $register_details['closed_at'] : \Carbon::now('GMT')->toDateTimeString();

        $is_types_of_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');

        $sell_no_registers = $this->cashRegisterUtil->getNoRegisterDetails($open_time, $close_time);

        $details = $this->cashRegisterUtil->getRegisterTransactionDetails($user_id, $id, $open_time, $close_time, $is_types_of_service_enabled);

        $payment_types = $this->cashRegisterUtil->payment_types($register_details->location_id, true, $business_id);

        $pos_settings = ! empty(request()->session()->get('business.pos_settings')) ? json_decode(request()->session()->get('business.pos_settings'), true) : [];
        
        return view('cash_register.close_register_modal')
                    ->with(compact('register_details', 'details', 'payment_types', 'pos_settings', 'id', 'sell_no_registers'));
    }

    /**
     * Closes currently opened register.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function postCloseRegister(Request $request)
    {        
        if (! auth()->user()->can('close_cash_register')) {
            abort(403, 'Unauthorized action.');
        }
    
        // dd($request->all());
        try {
            //Disable in demo
            if (config('app.env') == 'demo') {
                $output = ['success' => 0,
                    'msg' => 'Feature disabled in demo!!',
                ];

                return redirect()->action([\App\Http\Controllers\HomeController::class, 'index'])->with('status', $output);
            }

            $input = $request->only(['closing_amount', 'total_card_slips', 'total_cheques', 'closing_note']);
            $input['closing_amount'] = $this->cashRegisterUtil->num_uf($input['closing_amount']);
            $user_id = $request->input('user_id');
            $input['closed_at'] = \Carbon::now('GMT')->format('Y-m-d H:i:s');
            $input['status'] = 'close';
            $input['denominations'] = ! empty(request()->input('denominations')) ? json_encode(request()->input('denominations')) : null;

            CashRegister::where('user_id', $user_id)
                                ->where('cash_register_list_id', $request->cash_register_list_id)
                                ->where('status', 'open')
                                ->update($input);                        

            // dd('masuk');

            $business_id = request()->session()->get('user.business_id');

            $cash_register_list_id = $request->get('cash_register_list_id');        

            $register_details = $this->cashRegisterUtil->getRegisterDetails($request->get('id_cash_register'));

            $user_id = auth()->user()->id;
            $open_time = $register_details['open_time'];
            // $close_time = \Carbon::now('GMT')->toDateTimeString();
            $close_time = $register_details['closed_at'] ? $register_details['closed_at'] : '';

            $is_types_of_service_enabled = $this->moduleUtil->isModuleEnabled('types_of_service');

            $sell_no_registers = $this->cashRegisterUtil->getNoRegisterDetails($open_time, $close_time);

            $details = $this->cashRegisterUtil->getRegisterTransactionDetails($user_id, $cash_register_list_id, $open_time, $close_time, $is_types_of_service_enabled);

            $payment_types = $this->cashRegisterUtil->payment_types($register_details->location_id, true, $business_id);

            // return view('report.z_report_details')
                    // ->with(compact('register_details', 'details', 'payment_types', 'close_time'));
           
            $output = ['success' => 1,
                'msg' => __('cash_register.close_success'),
                'receipt' => [
                    'data' => [],
                    'is_enabled' => true,
                    'print_type' => 'browser',
                    'printer_config' => [],
                    'printer_title' => 'z-report-details',
                    'html_content' => view('report.z_report_details', compact('register_details', 'details', 'payment_types', 'close_time', 'sell_no_registers'))->render()   
                ]
            ]; 

            // dd($output);

            // dd($output);
            // $this->performLogout($request);
            // return redirect()->route('login')->with();
            // return redirect()->back()->with('status', $output);
            if($request->post_type == 'pos') {
                return $output;
            } else {
                return redirect()->back()->with('status', $output);
            }
        } catch (\Exception $e) {
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            // dd($e);
            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
            // dd($output);
            // return redirect()->back()->with('status', $output);
            if($request->segment(1) == 'pos') {
                return $output;
            } else {
                return redirect()->back()->with('status', $output);
            }
        }

        // return redirect()->action([\App\Http\Controllers\ReportController::class, 'create'], ['sub_type' => $sub_type])
    }

    public function getListCashRegister($id)
    {
        $business_id = request()->session()->get('user.business_id');
        $cr = CashRegisterList::where('business_id', $business_id)->where('location_id', $id)->pluck('id', 'name');
        // array_splice($cr, 0, 0, array(null => 'Please Select'));        
        return $cr;
    }
}
