<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\Contact;
use App\Events\TransactionPaymentDeleted;
use App\Http\Controllers\PostCaller;
use App\Transaction;
use App\TransactionPayment;
use App\TransactionSellLine;
use App\User;
use App\Utils\BusinessUtil;
use App\Utils\CashRegisterUtil;
use App\Utils\ContactUtil;
use App\Utils\ModuleUtil;
use App\Utils\NotificationUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Activitylog\Models\Activity;
use Yajra\DataTables\Facades\DataTables;
use Session;

class SellReturnController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $productUtil;

    protected $transactionUtil;

    protected $contactUtil;

    protected $businessUtil;

    protected $moduleUtil;

    protected $cashRegisterUtil;

    protected $notificationUtil;

    /**
     * Constructor
     *
     * @param  ProductUtils  $product
     * @return void
     */
     
     
    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, ContactUtil $contactUtil, BusinessUtil $businessUtil, ModuleUtil $moduleUtil, CashRegisterUtil $cashRegisterUtil, NotificationUtil $notificationUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->contactUtil = $contactUtil;
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;
        $this->cashRegisterUtil = $cashRegisterUtil;
        $this->notificationUtil = $notificationUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        if (request()->ajax()) {
            $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')

                    ->join(
                        'business_locations AS bl',
                        'transactions.location_id',
                        '=',
                        'bl.id'
                    )
                    ->join(
                        'transactions as T1',
                        'transactions.return_parent_id',
                        '=',
                        'T1.id'
                    )
                    ->leftJoin(
                        'transaction_payments AS TP',
                        'transactions.id',
                        '=',
                        'TP.transaction_id'
                    )
                    ->where('transactions.business_id', $business_id)
                    ->where('transactions.type', 'sell_return')
                    ->where('transactions.status', 'final')
                    ->select(
                        'transactions.id',
                        'transactions.transaction_date',
                        'transactions.invoice_no',
                        'contacts.name',
                        'contacts.supplier_business_name',
                        'transactions.final_total',
                        'transactions.payment_status',
                        'bl.name as business_location',
                        'T1.invoice_no as parent_sale',
                        'T1.id as parent_sale_id',
                        DB::raw('SUM(TP.amount) as amount_paid')
                    );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
                $sells->where('transactions.created_by', request()->session()->get('user.id'));
            }

            //Add condition for created_by,used in sales representative sales report
            if (request()->has('created_by')) {
                $created_by = request()->get('created_by');
                if (! empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            //Add condition for location,used in sales representative expense report
            if (request()->has('location_id')) {
                $location_id = request()->get('location_id');
                if (! empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if (! empty(request()->customer_id)) {
                $customer_id = request()->customer_id;
                $sells->where('contacts.id', $customer_id);
            }
            if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                $start = request()->start_date;
                $end = request()->end_date;
                $sells->whereDate('transactions.transaction_date', '>=', $start)
                        ->whereDate('transactions.transaction_date', '<=', $end);
            }

            $sells->groupBy('transactions.id');

            return Datatables::of($sells)
                ->addColumn(
                    'action',
                    '<div class="btn-group">
                    <button type="button" class="btn btn-info dropdown-toggle btn-xs" 
                        data-toggle="dropdown" aria-expanded="false">'.
                        __('messages.actions').
                        '<span class="caret"></span><span class="sr-only">Toggle Dropdown
                        </span>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-right" role="menu">
                        <li><a href="#" class="btn-modal" data-container=".view_modal" data-href="{{action(\'App\Http\Controllers\SellReturnController@show\', [$parent_sale_id])}}"><i class="fas fa-eye" aria-hidden="true"></i> @lang("messages.view")</a></li>
                        {{-- <li><a href="{{action(\'App\Http\Controllers\SellReturnController@add\', [$parent_sale_id])}}" ><i class="fa fa-edit" aria-hidden="true"></i> @lang("messages.edit")</a></li> --}}
                        <li><a href="{{action(\'App\Http\Controllers\SellReturnController@destroy\', [$id])}}" class="delete_sell_return" ><i class="fa fa-trash" aria-hidden="true"></i> @lang("messages.delete")</a></li>
                        <li><a href="#" class="print-invoice" data-href="{{action(\'App\Http\Controllers\SellReturnController@printInvoice\', [$id])}}"><i class="fa fa-print" aria-hidden="true"></i> @lang("messages.print")</a></li>

                    @if($payment_status != "paid")
                        <li><a href="{{action(\'App\Http\Controllers\TransactionPaymentController@addPayment\', [$id])}}" class="add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a></li>
                    @endif

                    <li><a href="{{action(\'App\Http\Controllers\TransactionPaymentController@show\', [$id])}}" class="view_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.view_payments")</a></li>
                    </ul>
                    </div>'
                )
                ->removeColumn('id')
                ->editColumn(
                    'final_total',
                    '<span class="display_currency final_total" data-currency_symbol="true" data-orig-value="{{$final_total}}">{{$final_total}}</span>'
                )
                ->editColumn('parent_sale', function ($row) {
                    return '<button type="button" class="btn btn-link btn-modal" data-container=".view_modal" data-href="'.action([\App\Http\Controllers\SellController::class, 'show'], [$row->parent_sale_id]).'">'.$row->parent_sale.'</button>';
                })
                ->editColumn('name', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$name}}')
                ->editColumn('transaction_date', '{{@format_datetime($transaction_date)}}')
                ->editColumn(
                    'payment_status',
                    '<a href="{{ action([\App\Http\Controllers\TransactionPaymentController::class, \'show\'], [$id])}}" class="view_payment_modal payment-status payment-status-label" data-orig-value="{{$payment_status}}" data-status-name="{{__(\'lang_v1.\' . $payment_status)}}"><span class="label @payment_status($payment_status)">{{__(\'lang_v1.\' . $payment_status)}}</span></a>'
                )
                ->addColumn('payment_due', function ($row) {
                    $due = $row->final_total - $row->amount_paid;

                    return '<span class="display_currency payment_due" data-currency_symbol="true" data-orig-value="'.$due.'">'.$due.'</sapn>';
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        if (auth()->user()->can('sell.view')) {
                            return  action([\App\Http\Controllers\SellReturnController::class, 'show'], [$row->parent_sale_id]);
                        } else {
                            return '';
                        }
                    }, ])
                ->rawColumns(['final_total', 'action', 'parent_sale', 'payment_status', 'payment_due', 'name'])
                ->make(true);
        }
        $business_locations = BusinessLocation::forDropdown($business_id, false);
        $customers = Contact::customersDropdown($business_id, false);

        $sales_representative = User::forDropdown($business_id, false, false, true);

        return view('sell_return.index')->with(compact('business_locations', 'customers', 'sales_representative'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    // public function create()
    // {
    //     if (!auth()->user()->can('sell.create')) {
    //         abort(403, 'Unauthorized action.');
    //     }

    //     $business_id = request()->session()->get('user.business_id');

    //     //Check if subscribed or not
    //     if (!$this->moduleUtil->isSubscribed($business_id)) {
    //         return $this->moduleUtil->expiredResponse(action([\App\Http\Controllers\SellReturnController::class, 'index']));
    //     }

    //     $business_locations = BusinessLocation::forDropdown($business_id);
    //     //$walk_in_customer = $this->contactUtil->getWalkInCustomer($business_id);

    //     return view('sell_return.create')
    //         ->with(compact('business_locations'));
    // }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function add($id)
    {                
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        //Check if subscribed or not
        if (! $this->moduleUtil->isSubscribed($business_id)) {
            return $this->moduleUtil->expiredResponse();
        }

        $sell = Transaction::where('transactions.business_id', $business_id)                            
                            ->with(['sell_lines', 'location', 'contact', 'tax', 'sell_lines.sub_unit', 'sell_lines.product', 'sell_lines.product.unit', 'payment_lines', 'return_parent'])                            
                            ->find($id);
        
        if($sell->payment_status != 'paid') {
            return redirect()->back()->with('errors', trans('lang_v1.sell_not_found').' or '.trans('lang_v1.view_paid_sells_only'));
        }

        $business_details = $this->businessUtil->getDetails($business_id);
        $qtySell = 0;
        $qtyReturn = 0;
        
        foreach ($sell->sell_lines as $key => $value) {
            // dd($value->payment_type);
            if (! empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }           
            $qtySell = $qtySell + $value->quantity;
            $qtyReturn = $qtyReturn + $value->quantity_returned;
            
            $sell->sell_lines[$key]->formatted_qty = $this->transactionUtil->num_f($value->quantity, false, null, true);
        }

        $disable_return = true;
        if($qtySell != $qtyReturn) {
            $disable_return = false;
        }

        $default_location = null;
        if($sell->location) {
            $default_location = $sell->location->id;
        }        
        // dd($sell->sell_lines)
        return view('sell_return.add')
        ->with(compact('sell', 'default_location', 'business_details', 'disable_return'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }        
        // dd($request->all());
        try {
            $input = $request->except('_token'); 
            // dd($input);                                                
            $business_id = $request->session()->get('user.business_id');                        
            //Check if subscribed or not
            if (! $this->moduleUtil->isSubscribed($business_id)) {
                return $this->moduleUtil->expiredResponse(action([\App\Http\Controllers\SellReturnController::class, 'index']));
            }

            $user_id = $request->session()->get('user.id');
            if($request->get('voidable') == 'true') {
                $return_total = $request->get('return_subtotal');
            } else {
                // dd(isset($input['exchange_checkbox']));
                $return_total = isset($input['exchange_checkbox']) ? $request->get('final_total') : $request->get('return_subtotal');            
            }            
            
            DB::beginTransaction();
            
            $paymentMethod = null;
            $amountCard = 0; 
            $amountCash = 0;  
            $goingToCard = false;          
            $amountCalculate = round($return_total,2);        
            $input['return_total'] = round($return_total,2);            
            $haveCard = false;
            if (! empty($input['return_products'])) {                
                foreach ($input['return_products'] as $v1) {
                    # code...
                    if($v1['quantity'] > 0) {
                        foreach ($v1['payment_method'] as $k2 => $v2) {
                            # code...
                            if($v2['method'] == 'CARD' || $v2['method'] == 'DEBIT') {
                                $haveCard = true;
                            }
                        }
                    }
                }                            
                foreach ($input['return_products'] as $key => $value) {
                    # code...      
                    if($key == 0 ) {
                        foreach ($value['payment_method'] as $k => $v) {
                            # code...                                                                                                                                
                            if($v['amount'] > 0) {
                                if($haveCard) {
                                    if($v['method'] == 'CASH') {                                       
                                        $amountCalculate = (float)$amountCalculate - (float)$v['amount'];                                       
                                        if($amountCalculate >= 0) {                                    
                                            $amountCash = (float)$v['amount']; 
                                        } else {                
                                            if($amountCalculate < 0) {
                                                $amountCash = (float)$v['amount'] - $amountCalculate; 
                                            } else {
                                                $amountCash = $amountCalculate;
                                            }
                                        }                    
                                        $input['return_products'][$key]['payment_method'][$k]['amount_used'] = $amountCash;                            
                                        
                                    }         
                                    
                                    if($v['method'] == 'CARD' && $amountCalculate > 0) {
                                        // dd((float)number_format($return_total,2), (float)$v['amount']);  
                                        $amountCalculate = (float)$amountCalculate - (float)$v['amount'];                                           
                                        if($amountCalculate >= 0) {                                    
                                            $amountCard = (float)$v['amount']; 
                                        } else {                
                                            if($amountCalculate < 0) {
                                                $amountCard = (float)$v['amount'] + $amountCalculate; 
                                            } else {
                                                $amountCard = $amountCalculate;
                                            }
                                        } 
                                        if($request->has('exchange_checkbox')) {
                                            $goingToCard = false;
                                        } else {
                                            $goingToCard = true;
                                        }
                                        $input['return_products'][$key]['payment_method'][$k]['amount_used'] = $amountCard;
                                        // var_dump('card', $amountCard, $amountCalculate);
                                    }        

                                    if($v['method'] == 'DEBIT' && $amountCalculate > 0) {
                                        // dd((float)number_format($return_total,2), (float)$v['amount']);  
                                        $amountCalculate = (float)$amountCalculate - (float)$v['amount'];                                           
                                        if($amountCalculate >= 0) {                                    
                                            $amountCard = (float)$v['amount']; 
                                        } else {                
                                            if($amountCalculate < 0) {
                                                $amountCard = (float)$v['amount'] + $amountCalculate; 
                                            } else {
                                                $amountCard = $amountCalculate;
                                            }
                                        } 
                                        if($request->has('exchange_checkbox')) {
                                            $goingToCard = false;
                                        } else {
                                            $goingToCard = true;
                                        }
                                        $input['return_products'][$key]['payment_method'][$k]['amount_used'] = $amountCard;
                                        // var_dump('card', $amountCard, $amountCalculate);
                                    }                                   
                                } else {
                                    $amountCash = $return_total;
                                }
                            } 
                            
                        }                                      
                    }              
                }
                
                $input['amount_cash'] = $amountCash;
                $input['amount_card'] = $amountCard;                       
                // dd($input['return_products'], $amountCard, $amountCash, $amountCalculate, $goingToCard);
                if($goingToCard) {
                    /* foreach ($value['payment_method'] as $k => $v) {
                        $amountCalculate+= $v['amount'];
                    } */
                                    
                    // $amountCard = $return_total;
                    /* if((float)number_format($request->get('return_subtotal'), 2) < (float)number_format($amountCalculate, 2)) {
                        return [
                            'success' => 0,
                            'msg' => "Return amount is not correct, please check the amount and try again",
                            'receipt' => null,
                        ];
                    } */
                    $ref_count = $this->transactionUtil->setAndGetReferenceCount('sell_return', $business_id);
                    $input['invoice_no'] = $this->transactionUtil->generateReferenceNumber('sell_return', $ref_count, $business_id);
                    $respText = null;
                    
                    $get_settings = DB::table('cardtransactions')
                        ->join('card_pointe_settings as cps', 'cps.merchant_id', '=', 'cardtransactions.merchid')
                        ->join('card_pointe_cash_registers as cpcr', 'cpcr.branch_id', '=', 'cps.branch_id')
                        ->select('cardtransactions.*', 'cps.merchant_authorization', 'cpcr.isv_url')
                        ->where('invoiceid', $input['invoice_no_pos'])->first();                                        
                    if($get_settings == null) {
                        return ['success' => 0,
                            'msg' => __('lang_v1.card_payment_not_found'),
                            'receipt' => null,
                        ];
                    } else {
                        $returnCard = $this->returnCardPayment($get_settings->retref, $get_settings->merchid, $get_settings->merchant_authorization, number_format($amountCard, 2), $get_settings->isv_url);                        
                        if($returnCard->respcode == '000' || $returnCard->respcode == '00' || $returnCard->respcode == '0') {                            
                            DB::table('cardtransactions')->insert([
                                'token' => $get_settings->token,
                                'expiry' => $get_settings->expiry,
                                'name' => $get_settings->name,
                                'retref' => $get_settings->retref,
                                'respproc' => $returnCard->respproc,
                                'amount' => '-'.$amountCard,
                                'resptext' => $returnCard->resptext,
                                'authcode' => $get_settings->authcode,
                                'respcode' => $returnCard->respcode,
                                'merchid' => $get_settings->merchid,
                                'PIN' => $get_settings->PIN,
                                'signature' => $get_settings->signature,
                                'networkLabel' => $get_settings->networkLabel,
                                'applicationPreferredName' => $get_settings->applicationPreferredName,
                                'AID' => $get_settings->AID,
                                'applicationLabel' => $get_settings->applicationLabel,
                                'invoiceid' => $input['invoice_no'].' From '.$get_settings->invoiceid,
                                'entrymode' => $get_settings->entrymode,
                                'country' => $get_settings->country,
                                'bin' => $get_settings->bin,
                                'issuer' => $get_settings->issuer
                            ]);                            
                        } else {                            
                            if(property_exists($returnCard, 'resptext') || array_key_exists('resptext',$returnCard)) {
                                DB::table('cardtransactions')->insert([
                                    'token' => $get_settings->token,
                                    'expiry' => $get_settings->expiry,
                                    'name' => $get_settings->name,
                                    'retref' => $get_settings->retref,
                                    'respproc' => property_exists($returnCard, 'respproc') ? $returnCard->respproc : null,
                                    'amount' => '-'.$amountCard,
                                    'resptext' => $returnCard->resptext,
                                    'authcode' => $get_settings->authcode,
                                    'respcode' => $returnCard->respcode,
                                    'merchid' => $get_settings->merchid,
                                    'PIN' => $get_settings->PIN,
                                    'signature' => $get_settings->signature,
                                    'networkLabel' => $get_settings->networkLabel,
                                    'applicationPreferredName' => $get_settings->applicationPreferredName,
                                    'AID' => $get_settings->AID,
                                    'applicationLabel' => $get_settings->applicationLabel,
                                    'invoiceid' => $input['invoice_no'].' From '.$get_settings->invoiceid,
                                    'entrymode' => $get_settings->entrymode,
                                    'country' => $get_settings->country,
                                    'bin' => $get_settings->bin,
                                    'issuer' => $get_settings->issuer
                                ]);
                                DB::commit();
                                return [
                                    'success' => 0,
                                    'msg' => $returnCard->resptext,
                                    'receipt' => null,
                                ];
                            } else {
                                DB::table('cardtransactions')->insert([
                                    'token' => $get_settings->token,
                                    'expiry' => $get_settings->expiry,
                                    'name' => $get_settings->name,
                                    'retref' => $get_settings->retref,
                                    'respproc' => null,
                                    'amount' => '-'.$amountCard,
                                    'resptext' => "Something wrong with card gateway",
                                    'authcode' => $get_settings->authcode,
                                    'respcode' => "9999",
                                    'merchid' => $get_settings->merchid,
                                    'PIN' => $get_settings->PIN,
                                    'signature' => $get_settings->signature,
                                    'networkLabel' => $get_settings->networkLabel,
                                    'applicationPreferredName' => $get_settings->applicationPreferredName,
                                    'AID' => $get_settings->AID,
                                    'applicationLabel' => $get_settings->applicationLabel,
                                    'invoiceid' => $input['invoice_no']. ' From '.$get_settings->invoiceid,
                                    'entrymode' => $get_settings->entrymode,
                                    'country' => $get_settings->country,
                                    'bin' => $get_settings->bin,
                                    'issuer' => $get_settings->issuer
                                ]);
                                DB::commit();
                                return [
                                    'success' => 0,
                                    'msg' => "Something wrong with card gateway",
                                    'receipt' => null,
                                ];
                            }
                        }                        
                    }

                    $sell_return = $this->transactionUtil->addSellReturn($input, $business_id, $user_id);
                    // $sell_return = null;
                    $cardReturnDone = true;
                    // dd($sell_return->invoice_no);
                } else {
                    /* foreach ($value['payment_method'] as $k => $v) {
                        $amountCalculate+= $v['amount'];
                    } */
                                            
                    /* if(number_format($request->get('return_subtotal'), 2) < $amountCalculate) {
                        return [
                            'success' => 0,
                            'msg' => "Return amount is not correct, please check the amount and try again",
                            'receipt' => null,
                        ];
                    } */                    
                    $sell_return = $this->transactionUtil->addSellReturn($input, $business_id, $user_id);                    
                    // dd($sell_return->invoice_no);
                }
                
                $receipt = $this->receiptContent($business_id, $sell_return->location_id, $sell_return->id);                
                
                if($request->has('exchange_checkbox')) {                        
                    if(isset($input['products'])) {
                        
                        // $transaction = Transaction::find($input['transaction_id']);
                        // dd($transaction);
                        $data = [
                            "location_id" => $input['location_id'],
                            "price_group" => "0",
                            "default_price_group" => null,
                            "contact_id" => $input['customer_id'],
                            "pay_term_number" => null,
                            "pay_term_type" => null,                    
                            "status" => "final",
                            "invoice_scheme_id" => "1",
                            "invoice_no" => null,
                            "search_product" => null,
                            "sell_price_tax" => "includes",
                            "products" => $input['products'],
                            "discount_type" => "percentage",
                            "discount_amount" => "0.00",
                            "rp_redeemed" => "0",
                            "rp_redeemed_amount" => "0",
                            "tax_rate_id" => $input['tax_id'],
                            "tax_calculation_amount" => $input['tax_calculation_amount'],
                            "sale_note" => "Exchange Transaction Invoice No: ".$input['invoice_no_pos'],
                            "is_direct_sale" => "1",
                            "shipping_details" => null,
                            "shipping_address" => null,
                            "shipping_charges" => "0.00",
                            "shipping_status" => null,
                            "delivered_to" => null,
                            "additional_expense_key_1" => null,
                            "additional_expense_value_1" => "0",
                            "additional_expense_key_2" => null,
                            "additional_expense_value_2" => "0",
                            "additional_expense_key_3" => null,
                            "additional_expense_value_3" => "0",
                            "additional_expense_key_4" => null,
                            "additional_expense_value_4" => "0",
                            "final_total" => $input['final_total'],
                            "advance_balance" => "0.0000",
                            "payment" => [
                                [
                                "amount" => $input['final_total'],                        
                                "method" => "cash",
                                "card_number" => null,
                                "card_holder_name" => null,
                                "card_transaction_number" => null,
                                "card_type" => "credit",
                                "card_month" => null,
                                "card_year" => null,
                                "card_security" => null,
                                "cheque_number" => null,
                                "bank_account_number" => null,
                                "transaction_no_1" => null,
                                "transaction_no_2" => null,
                                "transaction_no_3" => null,
                                "transaction_no_4" => null,
                                "transaction_no_5" => null,
                                "transaction_no_6" => null,
                                "transaction_no_7" => null,
                                ],
                                "change_return" => [
                                    "method" => "cash",
                                    "card_number" => null,
                                    "card_holder_name" => null,
                                    "card_transaction_number" => null,
                                    "card_type" => "credit",
                                    "card_month" => null,
                                    "card_year" => null,
                                    "card_security" => null,
                                    "cheque_number" => null,
                                    "bank_account_number" => null,
                                    "transaction_no_1" => null,
                                    "transaction_no_2" => null,
                                    "transaction_no_3" => null,
                                    "transaction_no_4" => null,
                                    "transaction_no_5" => null,
                                    "transaction_no_6" => null,
                                    "transaction_no_7" => null,
                                ]
                            ],
                            "change_return" => "0.00",
                            "is_save_and_print" => "0",
                            "recur_interval" => null,
                            "recur_interval_type" => "days",
                            "recur_repetitions" => null,
                            "subscription_repeat_on" => null,
                            "type" => "sell",
                            "sub_type" => "exchange",
                            "transfer_parent_id" => $input['transaction_id']           
                        ];
                        /* $post = new PostCaller(
                            SellPosController::class,
                            'store',
                            $request,
                            $data,
                        );                                                                       
                        $storeSellPos = $post->call(); */
                                                        
                        // dd($storeSellPos);
        
                        // dd($data);
                        $request->replace($data);                
                        $sellPosController = app(SellPosController::class)->store($request);                
                        if($sellPosController['success'] == 1) {
                            $receipt = $sellPosController['receipt'];
                            $transaction_id = $sellPosController['transactionid'];
                            $transaction_sell_return = Transaction::find($sell_return->id);
                            $transaction_sell_return->transfer_parent_id = $transaction_id;
                            $transaction_sell_return->save();
                        } else {
                            return [
                                'success' => 0,
                                'msg' => $sellPosController['msg'],
                                'receipt' => null
                            ];
                        }
                        // DB::commit();             
                        
        
                        /* $output = ['success' => 1,
                            'msg' => __('lang_v1.success'),
                            'receipt' => $receipt,
                        ];
                        return $output; */
                    } else {
                        return [
                            'success' => 0,
                            'msg' => 'No products found, please add one exchange product or more'
                        ];
                    }                
                }

            }
            
            DB::commit();
            $output = ['success' => 1,
                'msg' => __('lang_v1.success'),
                'receipt' => $receipt,
            ];
            
            return $output;

        } catch (\Exception $e) {
            DB::rollBack();   
            // dd($e);          
            if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                $msg = $e->getMessage();
            } else {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                $msg = __('messages.something_went_wrong');
            }

            $output = ['success' => 0,
                'msg' => $msg,
                'receipt' => null,
            ];
        }

        return $output;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $query = Transaction::where('business_id', $business_id)
                                ->where('id', $id)
                                ->with(
                                    'contact',
                                    'return_parent',
                                    'tax',
                                    'sell_lines',
                                    'sell_lines.product',
                                    'sell_lines.variations',
                                    'sell_lines.sub_unit',
                                    'sell_lines.product',
                                    'sell_lines.product.unit',
                                    'location'
                                );  
                                      
        if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
            $sells->where('created_by', request()->session()->get('user.id'));
        }
        $sell = $query->first();

        foreach ($sell->sell_lines as $key => $value) {
            if (! empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }
        }

        $get_payment = [];
        if(! empty($sell->return_parent)) {
            $get_payment = TransactionPayment::where('transaction_id', $sell->return_parent->id)->get();
        }

        $sell_taxes = [];
        if (! empty($sell->return_parent->tax)) {
            if ($sell->return_parent->tax->is_tax_group) {
                $sell_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($sell->return_parent->tax, $sell->return_parent->tax_amount));
            } else {
                $sell_taxes[$sell->return_parent->tax->name] = $sell->return_parent->tax_amount;
            }
        }

        $total_discount = 0;
        if ($sell->return_parent->discount_type == 'fixed') {
            $total_discount = $sell->return_parent->discount_amount;
        } elseif ($sell->return_parent->discount_type == 'percentage') {
            $discount_percent = $sell->return_parent->discount_amount;
            if ($discount_percent == 100) {
                $total_discount = $sell->return_parent->total_before_tax;
            } else {
                $total_after_discount = $sell->return_parent->final_total - $sell->return_parent->tax_amount;
                $total_before_discount = $total_after_discount * 100 / (100 - $discount_percent);
                $total_discount = $total_before_discount - $total_after_discount;
            }
        }
        

        $activities = Activity::forSubject($sell->return_parent)
           ->with(['causer', 'subject'])
           ->latest()
           ->get();

        return view('sell_return.show')
            ->with(compact('sell', 'sell_taxes', 'total_discount', 'activities', 'get_payment'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                //Begin transaction
                DB::beginTransaction();

                $query = Transaction::where('id', $id)
                    ->where('business_id', $business_id)
                    ->where('type', 'sell_return')
                    ->with(['sell_lines', 'payment_lines']);

                if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
                    $sells->where('created_by', request()->session()->get('user.id'));
                }
                $sell_return = $query->first();

                $sell_lines = TransactionSellLine::where('transaction_id',
                                            $sell_return->return_parent_id)
                                    ->get();

                if (! empty($sell_return)) {
                    $transaction_payments = $sell_return->payment_lines;

                    foreach ($sell_lines as $sell_line) {
                        if ($sell_line->quantity_returned > 0) {
                            $quantity = 0;
                            $quantity_before = $this->transactionUtil->num_f($sell_line->quantity_returned);

                            $sell_line->quantity_returned = 0;
                            $sell_line->save();

                            //update quantity sold in corresponding purchase lines
                            $this->transactionUtil->updateQuantitySoldFromSellLine($sell_line, 0, $quantity_before);

                            // Update quantity in variation location details
                            $this->productUtil->updateProductQuantity($sell_return->location_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before, null, true,  null, null, null);
                        }
                    }

                    $sell_return->delete();
                    foreach ($transaction_payments as $payment) {
                        event(new TransactionPaymentDeleted($payment));
                    }
                }

                DB::commit();
                $output = ['success' => 1,
                    'msg' => __('lang_v1.success'),
                ];
            } catch (\Exception $e) {
                DB::rollBack();

                if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                    $msg = $e->getMessage();
                } else {
                    \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                    $msg = __('messages.something_went_wrong');
                }

                $output = ['success' => 0,
                    'msg' => $msg,
                ];
            }

            return $output;
        }
    }

    /**
     * Returns the content for the receipt
     *
     * @param  int  $business_id
     * @param  int  $location_id
     * @param  int  $transaction_id
     * @param  string  $printer_type = null
     * @return array
     */
    private function receiptContent(
        $business_id,
        $location_id,
        $transaction_id,
        $printer_type = null
    ) {
        $output = ['is_enabled' => false,
            'print_type' => 'browser',
            'html_content' => null,
            'printer_config' => [],
            'data' => [],
        ];

        $business_details = $this->businessUtil->getDetails($business_id);
        $location_details = BusinessLocation::find($location_id);

        //Check if printing of invoice is enabled or not.
        if ($location_details->print_receipt_on_invoice == 1) {
            //If enabled, get print type.
            $output['is_enabled'] = true;

            $invoice_layout = $this->businessUtil->invoiceLayout($business_id, $location_details->invoice_layout_id);

            //Check if printer setting is provided.
            $receipt_printer_type = is_null($printer_type) ? $location_details->receipt_printer_type : $printer_type;

            $receipt_details = $this->transactionUtil->getReceiptDetails($transaction_id, $location_id, $invoice_layout, $business_details, $location_details, $receipt_printer_type);
            // dd($receipt_details);
            //If print type browser - return the content, printer - return printer config data, and invoice format config
            $output['print_title'] = $receipt_details->invoice_no;
            if ($receipt_printer_type == 'printer') {
                $output['print_type'] = 'printer';
                $output['printer_config'] = $this->businessUtil->printerConfig($business_id, $location_details->printer_id);
                $output['data'] = $receipt_details;
            } else {
                // $layout = ! empty($receipt_details->design) ? 'sale_pos.receipts.'.$receipt_details->design : 'sale_pos.receipts.classic';

                // $output['html_content'] = view($layout, compact('receipt_details'))->render();
                $output['html_content'] = view('sell_return.new_receipt', compact('receipt_details'))->render();
            }
        }

        return $output;
    }

    /**
     * Prints invoice for sell
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function printInvoice(Request $request, $transaction_id)
    {
        if (request()->ajax()) {
            try {
                $output = ['success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];

                $business_id = $request->session()->get('user.business_id');

                $transaction = Transaction::where('business_id', $business_id)
                                ->where('id', $transaction_id)
                                ->first();

                if (empty($transaction)) {
                    return $output;
                }

                $receipt = $this->receiptContent($business_id, $transaction->location_id, $transaction_id, 'browser');

                if (! empty($receipt)) {
                    $output = ['success' => 1, 'receipt' => $receipt];
                }
            } catch (\Exception $e) {    
                // dd($e);            
                $output = ['success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Function to validate sell for sell return
     */
    public function validateInvoiceToReturn($invoice_no)
    {
        if (! auth()->user()->can('sell.create') && ! auth()->user()->can('direct_sell.access') && ! auth()->user()->can('view_own_sell_only')) {
            return ['success' => 0,
                'msg' => trans('lang_v1.permission_denied'),
            ];
        }
        $business_id = request()->session()->get('user.business_id');
        $query = Transaction::where('business_id', $business_id)
        ->where('invoice_no', $invoice_no)
        ->where('payment_status', 'paid');        
        
        $permitted_locations = auth()->user()->permitted_locations();        
        if ($permitted_locations != 'all') {
            $query->whereIn('transactions.location_id', $permitted_locations);
        }

        /* if (! auth()->user()->can('direct_sell.access') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', auth()->user()->id);
        } */

        $sell = $query->first();
        // dd($sell);
        if (empty($sell)) {
            return ['success' => 0,
                'msg' => trans('lang_v1.sell_not_found').' or '.trans('lang_v1.view_paid_sells_only'),
            ];
        }

        return ['success' => 1,
            'redirect_url' => action([\App\Http\Controllers\SellReturnController::class, 'add'], [$sell->id]),
        ];
    }

    public function returnCardPayment($retref, $merchantId, $merchant_authorization, $amount, $isv_url) 
    {                
        // dd($retref, $merchantId, $merchant_authorization, $amount, $isv_url);
        if (!empty($retref) && !empty($merchantId) && !empty($merchant_authorization) && !empty($amount) && !empty($isv_url)) {
            $retref = $retref;
            $merchantId = $merchantId;
            $merchant_authorization = $merchant_authorization;                        
            $amount = $amount < 0 ? ($amount * -1) : $amount;
            $isv_url = $isv_url;                        
        }else {
            $notfound = '{"respcode":"3r0rr", "resptext": "Return card payment have incorrect data!"}';     
            return json_decode($notfound);
        }        
        
        try {

            // $curl = curl_init();

            // curl_setopt_array($curl, array(
            //     CURLOPT_URL => $isv_url . '/refund',
            //     CURLOPT_RETURNTRANSFER => true,
            //     CURLOPT_ENCODING => '',
            //     CURLOPT_MAXREDIRS => 10,
            //     CURLOPT_TIMEOUT => 0,
            //     CURLOPT_FOLLOWLOCATION => true,
            //     CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            //     CURLOPT_CUSTOMREQUEST => 'PUT',
            //     CURLOPT_POSTFIELDS =>'{
            //     "retref": "'.$retref.'",
            //     "merchid": "'.$merchantId.'",
            //     "amount": "'.$amount.'"
            //     }',
            //     CURLOPT_HTTPHEADER => array(
            //         'Content-Type: application/json',
            //         'Authorization: ' . $merchant_authorization
            //     ),
            // ));

            // $response = curl_exec($curl);
            // $response = json_decode($response);
            // dd($response);
            // curl_close($curl);
            // echo $response;

            $curl = curl_init();
            curl_setopt_array($curl, array(
                CURLOPT_URL => $isv_url . '/inquire/' . $retref . '/' . $merchantId,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                    'Authorization: ' . $merchant_authorization
                ),
            ));

            $response = curl_exec($curl); 
            curl_close($curl);

            //if refundable -> refund, if non refundable -> void. 
            $result = json_decode($response);            
            if(property_exists($result, 'error')) {
                $result = '{"respcode":"404", "resptext": "Error inquiry transaction!"}';
                return json_decode($result);
            } else {
                if($result->refundable == "Y")
                {                
                    $curl = curl_init();
                    curl_setopt_array($curl, array(
                        CURLOPT_URL => $isv_url . '/refund',
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_ENCODING => '',
                        CURLOPT_MAXREDIRS => 10,
                        CURLOPT_TIMEOUT => 0,
                        CURLOPT_FOLLOWLOCATION => true,
                        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                        CURLOPT_CUSTOMREQUEST => 'PUT',
                        CURLOPT_POSTFIELDS =>'{
                        "retref": "'.$retref.'",
                        "merchid": "'.$merchantId.'",
                        "amount": "'.$amount.'"
                        }',
                        CURLOPT_HTTPHEADER => array(
                            'Content-Type: application/json',
                            'Authorization: ' . $merchant_authorization
                        )
                    ));
                    $resp = curl_exec($curl);                
                    curl_close($curl);
                    return json_decode($resp);
                }
    
                if($result->refundable == "N")
                {
                    if($result->voidable == "Y")
                    {
                        $curl = curl_init();
                        curl_setopt_array($curl, array(
                            CURLOPT_URL => $isv_url . '/void',
                            CURLOPT_RETURNTRANSFER => true,
                            CURLOPT_ENCODING => '',
                            CURLOPT_MAXREDIRS => 10,
                            CURLOPT_TIMEOUT => 0,
                            CURLOPT_FOLLOWLOCATION => true,
                            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                            CURLOPT_CUSTOMREQUEST => 'PUT',
                            CURLOPT_POSTFIELDS =>'{
                            "retref": "'.$retref.'",
                            "merchid": "'.$merchantId.'",
                            "amount": "'.$amount.'"
                            }',
                            CURLOPT_HTTPHEADER => array(
                                'Content-Type: application/json',
                                'Authorization: ' . $merchant_authorization
                            )
                        ));
                        $resp = curl_exec($curl);
                        curl_close($curl);                    
                        return json_decode($resp);                    
                    }
                    else //Means that we cannot refund or void
                    {
                        $result = '{"respcode":"CCC", "resptext": "Non Refundable and Non Voidable!"}';
                        return json_decode($result);
                    }
                }
            }

        }catch (Exception $ex) {
            // echo $ex->getMessage();
            $result = '{"respcode":"CCC", "resptext": "'.$ex->getMessage().'!"}';
            return json_decode($result);
        }            
    }

    public function inquiryCardPayment($invoice_no_pos)
    {
        $get_settings = DB::table('cardtransactions')
                ->join('card_pointe_settings as cps', 'cps.merchant_id', '=', 'cardtransactions.merchid')
                ->join('card_pointe_cash_registers as cpcr', 'cpcr.branch_id', '=', 'cps.branch_id')
                ->select('cardtransactions.*', 'cps.merchant_authorization', 'cpcr.isv_url')
                ->where('invoiceid', $invoice_no_pos)->first();        
        if($get_settings) {
            $retref = $get_settings->retref;
            // $retref = '183040484494';
            $merchantId = $get_settings->merchid;
            $merchant_authorization = $get_settings->merchant_authorization;                                
            $isv_url = $get_settings->isv_url; 
    
            $curl = curl_init();
            curl_setopt_array($curl, array(
                CURLOPT_URL => $isv_url . '/inquire/' . $retref . '/' . $merchantId,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                    'Authorization: ' . $merchant_authorization
                ),
            ));
    
            $response = curl_exec($curl); 
            curl_close($curl);
    
            //if refundable -> refund, if non refundable -> void. 
            $result = json_decode($response);  
            // dd($result);        
            return $result;
        } else {
            return response()->json([
                'refundable' => 'Y',
                'voidable' => 'N'
            ]);
        }
    }
}
